<?php
// Disable caching for WSDL logic
ini_set("soap.wsdl_cache_enabled", "0");
ini_set('soap.wsdl_cache_ttl', '0'); 

class InventoryService {

    /**
     * The main SOAP function called by the client.
     * It receives an object structure based on the WSDL input.
     */
    public function StockLevel($request) {
        // 1. Extract UPC from the request object
        // Standard SoapServer converts <UPC>string</UPC> to $request->UPC
        $upcToFind = isset($request->UPC) ? trim($request->UPC) : null;

        // 2. Lookup the UPC in the CSV file
        $result = $this->lookupByUPC($upcToFind);

        // 3. Format the response according to "StockLevelResult" in WSDL
        return [
            "StockLevelResult" => [
                "StockLevel"    => $result['stock_level'],
                "DateAvailable" => $result['date_available']
            ]
        ];
    }

    /**
     * Helper function to parse CSV and find the item
     */
    private function lookupByUPC($searchUPC) {
        $filename = 'data.csv';

        // Default response for Invalid/Not Found
        $response = [
            'stock_level' => 'INVALID',
            'date_available' => ''
        ];

        if (!$searchUPC) {
            return $response;
        }

        if (!file_exists($filename)) {
            return $response; 
        }

        if (($handle = fopen($filename, "r")) !== FALSE) {
            $row = 0;
            while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                $row++;
                if($row == 1) continue; // Skip header

                // CSV: [0]=>UPC, [1]=>QTY, [2]=>Date, [3]=>SKU
                $csvUPC = trim($data[0]);
                $csvQty = trim($data[1]);
                $csvDate = trim($data[2]);

                if ($csvUPC === $searchUPC) {
                    
                    $quantityInt = intval($csvQty);

                    // --- LOGIC UPDATE HERE ---
                    if ($quantityInt <= 0) {
                        // Case: Out of Stock
                        $response['stock_level'] = "0"; 
                        $response['date_available'] = $csvDate;
                    } elseif ($quantityInt >= 100) {
                        // Case: 100 or more in stock
                        // Spec requires we return "100" to indicate "100+"
                        $response['stock_level'] = "100";
                        $response['date_available'] = "";
                    } else {
                        // Case: Between 1 and 99
                        $response['stock_level'] = (string)$quantityInt;
                        $response['date_available'] = "";
                    }
                    // -------------------------
                    
                    fclose($handle);
                    return $response;
                }
            }
            fclose($handle);
        }

        return $response;
    }
}

// Initialize the SOAP Server
// Point to the 'stock.wsdl' file we created.
$wsdlPath = "stock.wsdl";

try {
    $server = new SoapServer($wsdlPath);
    $server->setClass("InventoryService");
    $server->handle();
} catch (Exception $e) {
    // Basic error logging
    file_put_contents('php://stderr', $e->getMessage());
}
?>